const Wishlist = require('../Model').wishlists
const User = require('../Model').Users
const Category = require('../Model').Categories
const Product = require('../Model').Products
const Productimage = require('../Model').ProductImages
const fs = require('fs')
const slugify = require('slugify');

exports.AddProduct = async (req, res) => {
    try {
        const { title, quantity, category, discountamount, price, maincategory, content, discount } = req.body
        const { img1, img2, img3, img4, img5 } = req.files
        const images = []
        if (!title) return res.json({ status: 400, msg: `Title is required.` })
        if (!quantity) return res.json({ status: 400, msg: `Quantity is required.` })
        if (!maincategory) return res.json({ status: 400, msg: `Category is required.` })
        if (!price) return res.json({ status: 400, msg: `Price is required.` })
        if (!content) return res.json({ status: 400, msg: `Content is required.` })

        const productSlug = slugify(title, { replacement: '_', lower: true, strict: true })
        const newProduct = { title, quantity, category, discountamount, maincategory, price, content, discount, slug: productSlug }
        const prod = await Product.create(newProduct)

        const productDir = './public/products/'
        if (!fs.existsSync(productDir)) {
            fs.mkdirSync(productDir)
        }

        images.push(img1, img2, img3, img4, img5)
        images.map(async (data, i) => {
            if (data) {
                const date = new Date()
                const tag = `img${i + 1}`
                const fileName = `product_${Math.floor(Math.random() * 10000000)}_${productSlug}_${date.getTime()}.webp`
                const newImg = { product: prod.id, name: fileName, tag }
                await Productimage.create(newImg)
                await data.mv(`${productDir}/${fileName}`)
            }
        })
        return res.json({ status: 200, msg: `Product Uploaded!.` })
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` })
    }
}

exports.UpdateProduct = async (req, res) => {
    try {
        const { title, quantity, category, discountamount, productid, maincategory, price, content, discount } = req.body
        const images = []
        if (!title) return res.json({ status: 400, msg: `Title is required.` })
        if (!quantity) return res.json({ status: 400, msg: `Quantity is required.` })
        if (!maincategory) return res.json({ status: 400, msg: `Category is required.` })
        if (!price) return res.json({ status: 400, msg: `Price is required.` })
        if (!content) return res.json({ status: 400, msg: `Content is required.` })

        const productSlug = slugify(title, { replacement: '_', lower: true, strict: true }) // Updated
        const prod = await Product.findOne({
            where: { id: productid },
            include: [{ model: Productimage, as: 'images' }],
        })

        if (!prod) return res.json({ status: 400, msg: `Product not found!...` })

        prod.title = title
        prod.slug = productSlug
        prod.quantity = quantity
        prod.category = category
        prod.discountamount = discountamount
        prod.price = price
        prod.content = content
        prod.discount = discount
        await prod.save()

        // if product directory doesnt exists then create one
        const productDir = './public/products/'
        if (!fs.existsSync(productDir)) {
            fs.mkdirSync(productDir)
        }

        const img1 = !req?.files?.img1 ? null : req.files.img1
        const img2 = !req?.files?.img2 ? 0 : req.files.img2
        const img3 = !req?.files?.img3 ? 0 : req.files.img3
        const img4 = !req?.files?.img4 ? 0 : req.files.img4
        const img5 = !req?.files?.img5 ? 0 : req.files.img5

        images.push(img1, img2, img3, img4, img5)
        images.map(async (data, i) => {
            if (data && data !== 0) {
                const date = new Date()
                const tag = `img${i + 1}`
                const fileName = `product_${Math.floor(Math.random() * 10000000)}_${productSlug}_${date.getTime()}.webp`
                const newImg = { product: prod.id, name: fileName, tag }
                await Productimage.create(newImg)
                await data.mv(`${productDir}/${fileName}`)
            }
        })

        return res.json({ status: 200, msg: `Product Updated Successfully` })
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` })
    }
}

exports.FetchProductForUpdating = async (req, res) => {
    try {
        const { id } = req.params
        const product = await Product.findOne({
            where: { id: id },
            include: [
                { model: Category, as: 'cart' },
                { model: Productimage, as: 'images' },
            ],
        })
        if (!product) return res.json({ status: 400, msg: `Product not found!.` })

        return res.json({ status: 200, product, })
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` })
    }
}

exports.DeleteProduct = async (req, res) => {
    try {
        const { id } = req.body;
        const product = await Product.findOne({ where: { id: id }, })
        if (!product) return res.json({ status: 400, msg: `Product does not exists!...` })

        await product.destroy()

        return res.json({ status: 200, msg: `Product Successfully Deleted!...` });
    } catch (error) {
        return res.json({ status: 400, msg: `error ${error}` });
    }
};

exports.AllProduct = async (req, res) => {
    try {
        const items = await Product.findAll({
            order: [['createdAt', 'DESC']],
            include: [
                { model: Category, as: 'cart' },
                { model: Productimage, as: 'images' },
            ],
        });

        const prices = items.map(item => parseFloat(item.price));
        const minPrice = Math.min(...prices);
        const maxPrice = Math.max(...prices);

        return res.json({ status: 200, msg: items, minPrice, maxPrice });
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` });
    }
};

exports.RandomProducts = async (req, res) => {
    try {
        const count = await Product.count(); // Get the total number of products
        if (count === 0) return res.json({ status: 404, msg: `No products available!` });

        const randomIndexes = [];
        const limit = Math.min(5, count); // Ensure we don't fetch more than the available products

        while (randomIndexes.length < limit) {
            const randomIndex = Math.floor(Math.random() * count);
            if (!randomIndexes.includes(randomIndex)) {
                randomIndexes.push(randomIndex);
            }
        }

        const randomProducts = await Promise.all(
            randomIndexes.map(async (index) => {
                return await Product.findOne({
                    offset: index,
                    include: [
                        { model: Category, as: 'cart' },
                        { model: Productimage, as: 'images' },
                    ],
                });
            })
        );

        return res.json({ status: 200, msg: randomProducts });
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` });
    }
};

exports.AllProductsByCategory = async (req, res) => {
    try {
        const { cart } = req.params;
        const items = await Product.findAll({
            where: { category: cart },
            order: [['createdAt', 'DESC']],
            include: [
                { model: Category, as: 'cart' },
                { model: Productimage, as: 'images' },
            ],
        });

        const prices = items.map(item => parseFloat(item.price));
        const minPrice = Math.min(...prices);
        const maxPrice = Math.max(...prices);

        const item = await Category.findOne({ where: { slug: cart } });
        if (!item) return res.json({ status: 404, msg: `Category not found` });

        return res.json({ status: 200, msg: items, category: item, minPrice, maxPrice });
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` });
    }
};

exports.SingleProduct = async (req, res) => {
    try {
        const { id } = req.params
        const item = await Product.findOne({
            where: { id: id },
            include: [
                { model: Category, as: 'cart' },
                { model: Productimage, as: 'images' },
            ],
        })
        if (!item) return res.json({ status: 400, msg: `Product not found!.` })

        return res.json({ status: 200, msg: item })
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` })
    }
}

exports.DeleteProductImage = async (req, res) => {
    try {
        const { id } = req.body
        const findImage = await Productimage.findOne({
            where: { id: id },
        })
        if (!findImage) return res.json({ status: 400, msg: `Image does not exists!...` })

        const findPath = `./public/products/${findImage.name}`
        if (fs.existsSync(findPath)) {
            fs.unlinkSync(findPath)
        }
        await findImage.destroy()
        return res.json({ status: 200, msg: `Image Deleted!...`, imgs: findImage })
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` })
    }
}

exports.AddToWishlist = async (req, res) => {
    try {
        const { user, product, email, price, quantity } = req.body;
        // Check if user exists (should probably check Users, not Wishlist)
        let findUser = await User.findOne({ where: { email } });
        if (!findUser) return res.json({ status: 400, msg: `User not found!...` });

        // Check if this product is already in the user's wishlist
        const findWishlist = await Wishlist.findOne({ where: { product, user } });
        if (findWishlist) return res.json({ status: 400, msg: 'Product already in wishlist!' });

        // Add to wishlist
        const newCart = { product, user, price, quantity, email };
        await Wishlist.create(newCart);

        return res.json({ status: 200, msg: `Item Added to wishlist` });
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` });
    }
}
exports.RemoveWishlist = async (req, res) => {
    try {
        const { id } = req.body;
        const wishlist = await Wishlist.findOne({ where: { id: id }, })
        if (!wishlist) return res.json({ status: 400, msg: `Product does not exists!...` })

        await wishlist.destroy()
        return res.json({ status: 200, msg: `Item removed from wishlist` });

    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` })
    }
}
exports.GetWishlistByUser = async (req, res) => {
    try {
        const { user } = req.params;
        const items = await Wishlist.findAll({
            where: { user: user },
            order: [['createdAt', 'DESC']],
            include: [
                { 
                    model: Product, 
                    as: 'productDetails',
                    include: [
                        { model: Productimage, as: 'images' }
                    ]
                }
            ]
        });

        return res.json({ status: 200, msg: items });
    } catch (error) {
        return res.json({ status: 400, msg: `Error ${error}` })
    }
}